/*
 * Decompiled with CFR 0.152.
 */
package dev.jeka.core.api.kotlin;

import dev.jeka.core.api.depmanagement.JkDependencySet;
import dev.jeka.core.api.depmanagement.JkRepo;
import dev.jeka.core.api.depmanagement.JkRepoSet;
import dev.jeka.core.api.depmanagement.JkVersionedModule;
import dev.jeka.core.api.depmanagement.resolution.JkDependencyResolver;
import dev.jeka.core.api.depmanagement.resolution.JkResolveResult;
import dev.jeka.core.api.file.JkPathMatcher;
import dev.jeka.core.api.file.JkPathSequence;
import dev.jeka.core.api.java.JkJavaCompiler;
import dev.jeka.core.api.java.JkJavaProcess;
import dev.jeka.core.api.kotlin.JkKotlinJvmCompileSpec;
import dev.jeka.core.api.system.JkLog;
import dev.jeka.core.api.system.JkProcess;
import dev.jeka.core.api.system.JkProperties;
import dev.jeka.core.api.utils.JkUtilsAssert;
import dev.jeka.core.api.utils.JkUtilsPath;
import dev.jeka.core.api.utils.JkUtilsSystem;
import dev.jeka.core.api.utils.JkUtilsTime;
import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

public final class JkKotlinCompiler {
    public static final String KOTLIN_VERSION_OPTION = "jeka.kotlin.version";
    public static final JkPathMatcher KOTLIN_SOURCE_MATCHER = JkPathMatcher.of("**/*.kt", "*.kt");
    private static final String KOTLIN_HOME = "KOTLIN_HOME";
    private boolean failOnError = true;
    private boolean logOutput;
    private boolean logCommand;
    private final List<String> jvmOptions = new LinkedList<String>();
    private List<String> options = new LinkedList<String>();
    private JkRepoSet repos = JkRepoSet.of(JkRepo.ofLocal(), JkRepo.ofMavenCentral());
    private List<Plugin> plugins = new LinkedList<Plugin>();
    private JkPathSequence extraClasspath = JkPathSequence.of();
    private final String command;
    private final JarsVersionAndTarget jarsVersionAndTarget;
    private String cachedVersion;

    private JkKotlinCompiler(String command, JarsVersionAndTarget jarsVersionAndTarget) {
        this.command = command;
        this.jarsVersionAndTarget = jarsVersionAndTarget;
    }

    public static JkKotlinCompiler ofCommand(String command) {
        String effectiveCommand = command;
        if (JkUtilsSystem.IS_WINDOWS && !command.toLowerCase().endsWith(".bat")) {
            effectiveCommand = command + ".bat";
        }
        return new JkKotlinCompiler(effectiveCommand, null);
    }

    public static JkKotlinCompiler ofKotlinHomeCommand(String command) {
        String kotlinHome = System.getenv(KOTLIN_HOME);
        JkUtilsAssert.state(kotlinHome != null, "KOTLIN_HOME environment variable is not defined. Please define this environment variable in order to compile Kotlin sources.", new Object[0]);
        String commandPath = kotlinHome + File.separator + "bin" + File.separator + command;
        return JkKotlinCompiler.ofCommand(commandPath);
    }

    public static JkKotlinCompiler ofTarget(JkRepoSet repos, Target target, String kotlinVersion) {
        JkResolveResult resolveResult = JkDependencyResolver.of().addRepos(repos).resolve(JkDependencySet.of().and("org.jetbrains.kotlin:kotlin-compiler:" + kotlinVersion)).assertNoError();
        return new JkKotlinCompiler(null, new JarsVersionAndTarget(resolveResult.getFiles(), kotlinVersion, target));
    }

    public static JkKotlinCompiler ofJvm(JkRepoSet repos, String version) {
        return JkKotlinCompiler.ofTarget(repos, Target.JAVA, version);
    }

    public static JkKotlinCompiler ofJvm(JkRepoSet repos) {
        String version = JkProperties.get(KOTLIN_VERSION_OPTION);
        if (version == null) {
            JkLog.info("No jeka.kotlin.version specified, try to resolce Kotlin compiler on local machine", new Object[0]);
            return JkKotlinCompiler.ofKotlinHomeCommand("kotlinc");
        }
        JkLog.info("Kotlin JVM compiler resoled to version " + version, new Object[0]);
        return JkKotlinCompiler.ofJvm(repos, version);
    }

    public boolean isProvidedCompiler() {
        return this.command != null;
    }

    public String getVersion() {
        if (this.jarsVersionAndTarget != null) {
            return this.jarsVersionAndTarget.version;
        }
        if (this.cachedVersion != null) {
            return this.cachedVersion;
        }
        List<String> lines = JkProcess.of(this.command, "-version").execAndReturnOutput(new String[0]);
        String line = lines.get(0);
        this.cachedVersion = line.split(" ")[2].trim();
        return this.cachedVersion;
    }

    public Path getStdLib() {
        if (this.isProvidedCompiler()) {
            String value = System.getenv(KOTLIN_HOME);
            JkUtilsAssert.state(value != null, "KOTLIN_HOME environment variable is not defined.", new Object[0]);
            return Paths.get(value, new String[0]).resolve("lib/kotlin-stdlib.jar");
        }
        return this.repos.get("org.jetbrains.kotlin:kotlin-stdlib");
    }

    public JkPathSequence getStdJdk8Lib() {
        if (this.isProvidedCompiler()) {
            String value = System.getenv(KOTLIN_HOME);
            JkUtilsAssert.state(value != null, "KOTLIN_HOME environment variable is not defined.", new Object[0]);
            Path kotlinDir = Paths.get(value, new String[0]);
            return JkPathSequence.of().and(kotlinDir.resolve("lib/kotlin-stdlib.jar")).and(kotlinDir.resolve("lib/kotlin-stdlib-jdk7.jar")).and(kotlinDir.resolve("lib/kotlin-stdlib-jdk8.jar"));
        }
        return JkDependencyResolver.of().addRepos(this.repos).resolve(JkDependencySet.of("org.jetbrains.kotlin:kotlin-stdlib-jdk8:" + this.getVersion())).getFiles();
    }

    public JkKotlinCompiler setFailOnError(boolean fail) {
        this.failOnError = fail;
        return this;
    }

    public JkKotlinCompiler setLogCommand(boolean log) {
        this.logCommand = log;
        return this;
    }

    public JkKotlinCompiler setLogOutput(boolean log) {
        this.logOutput = log;
        return this;
    }

    public JkRepoSet getRepos() {
        return this.repos;
    }

    public JkKotlinCompiler setRepos(JkRepoSet repos) {
        this.repos = repos;
        return this;
    }

    public JkKotlinCompiler addJvmOption(String option) {
        this.jvmOptions.add(JkKotlinCompiler.toWindowsArg(option));
        return this;
    }

    public JkKotlinCompiler addPluginOption(String pluginId, String name, String value) {
        this.addOption("-P");
        this.addOption("plugin:" + pluginId + ":" + name + "=" + value);
        return this;
    }

    public JkKotlinCompiler addPlugin(Path pluginJar) {
        Plugin plugin = new Plugin();
        plugin.jar = pluginJar;
        this.plugins.add(plugin);
        return this;
    }

    public JkKotlinCompiler addPlugin(String pluginModule) {
        Plugin plugin = new Plugin();
        plugin.pluginModule = pluginModule.isEmpty() ? null : JkVersionedModule.of(pluginModule);
        this.plugins.add(plugin);
        return this;
    }

    public JkKotlinCompiler addOption(String option) {
        this.options.add(JkKotlinCompiler.toWindowsArg(option));
        return this;
    }

    public boolean compile(JkKotlinJvmCompileSpec compileSpec) {
        if (compileSpec.getSources().count(1, false) == 0) {
            JkLog.warn("No source to compile in " + compileSpec.getSources(), new Object[0]);
            return true;
        }
        Path outputDir = compileSpec.getOutputDir();
        List<String> effectiveOptions = compileSpec.getOptions();
        effectiveOptions.addAll(0, this.options);
        if (outputDir == null) {
            throw new IllegalStateException("Output dir option (-d) has not been specified on the compiler. Specified options : " + effectiveOptions);
        }
        JkUtilsPath.createDirectories(outputDir, new FileAttribute[0]);
        String message = "Compiling Kotlin sources " + compileSpec.getSources();
        if (JkLog.verbosity().isVerbose()) {
            message = message + " to " + outputDir + " using options : " + String.join((CharSequence)" ", effectiveOptions);
        }
        long start = System.nanoTime();
        JkLog.startTask(message, new Object[0]);
        Result result = this.run(compileSpec);
        JkLog.endTask("Done in " + JkUtilsTime.durationInMillis(start) + " milliseconds.");
        if (!result.success) {
            if (this.failOnError) {
                throw new IllegalStateException("Kotlin compiler failed " + result.params);
            }
            return false;
        }
        return true;
    }

    public List<String> getPlugins() {
        return this.plugins.stream().map(rec$ -> ((Plugin)rec$).toOption()).collect(Collectors.toList());
    }

    public List<String> getPluginOptions() {
        LinkedList<String> options = new LinkedList<String>();
        Iterator it = options.iterator();
        while (it.hasNext()) {
            String option = (String)it.next();
            if (!option.equals("-P") || !it.hasNext()) continue;
            options.add(option);
            options.add((String)it.next());
        }
        return Collections.unmodifiableList(options);
    }

    private Result run(JkKotlinJvmCompileSpec compileSpec) {
        JkProcess kotlincProcess;
        JkPathMatcher filter = KOTLIN_SOURCE_MATCHER.or(JkJavaCompiler.JAVA_SOURCE_MATCHER);
        LinkedList<String> sourcePaths = new LinkedList<String>();
        for (Path file : compileSpec.getSources().andMatcher(filter).getFiles()) {
            sourcePaths.add(file.toString());
        }
        if (sourcePaths.isEmpty()) {
            JkLog.warn("No Kotlin source found in " + compileSpec.getSources(), new Object[0]);
            return new Result(true, Collections.emptyList());
        }
        JkLog.info("" + sourcePaths.size() + " files to compile.", new Object[0]);
        JkLog.info("Kotlin version : " + this.getVersion() + ", Target JVM : " + compileSpec.getTargetVersion(), new Object[0]);
        LinkedList<String> loggedOptions = new LinkedList<String>(this.options);
        JkKotlinJvmCompileSpec effectiveSpec = compileSpec.copy();
        for (Plugin plugin : this.plugins) {
            effectiveSpec.addOptions(plugin.toOption());
            loggedOptions.add(plugin.toOption());
        }
        if (this.command != null) {
            JkLog.info("Use kotlin compiler : " + this.command + " with options " + loggedOptions, new Object[0]);
            kotlincProcess = JkProcess.of(this.command, new String[0]).addParams(this.jvmOptions.stream().map(JkKotlinCompiler::toJavaOption).collect(Collectors.toList()));
        } else {
            JkLog.info("Use kotlin compiler with options " + loggedOptions, new Object[0]);
            kotlincProcess = JkJavaProcess.ofJava("org.jetbrains.kotlin.cli.jvm.K2JVMCompiler").setClasspath(this.jarsVersionAndTarget.jars).addJavaOptions(this.jvmOptions).addParams("-no-stdlib", "-no-reflect");
        }
        ((JkProcess)((JkProcess)((JkProcess)((JkProcess)((JkProcess)kotlincProcess.addParams(JkKotlinCompiler.toWindowsArgs(effectiveSpec.getOptions()))).addParams(JkKotlinCompiler.toWindowsArgs(this.options))).addParams(JkKotlinCompiler.toWindowsArgs(sourcePaths))).setFailOnError(this.failOnError)).setLogCommand(this.logCommand)).setLogOutput(this.logOutput);
        int result = kotlincProcess.exec(new String[0]);
        return new Result(result == 0, kotlincProcess.getParams());
    }

    private static List<String> toWindowsArgs(List<String> args) {
        return args.stream().map(JkKotlinCompiler::toWindowsArg).collect(Collectors.toList());
    }

    private static String toWindowsArg(String arg) {
        if (!JkUtilsSystem.IS_WINDOWS) {
            return arg;
        }
        if (arg.startsWith("\"") && arg.endsWith("\"")) {
            return arg;
        }
        if (arg.contains(" ") || arg.contains(";") || arg.contains(",") || arg.contains("=")) {
            return '\"' + arg + '\"';
        }
        return arg;
    }

    private static String toJavaOption(String option) {
        String result = option.startsWith("-") ? option.substring(1) : option;
        return "-J" + result;
    }

    private class Plugin {
        Path jar;
        JkVersionedModule pluginModule;

        private Plugin() {
        }

        private Path getJar() {
            if (this.jar != null) {
                return this.jar;
            }
            this.jar = JkKotlinCompiler.this.repos.get(this.pluginModule);
            if (this.jar == null) {
                throw new IllegalStateException("Cannot retrieve module " + this.pluginModule);
            }
            return this.jar;
        }

        private String toOption() {
            return "-Xplugin=" + this.getJar();
        }
    }

    private static class JarsVersionAndTarget {
        final JkPathSequence jars;
        final String version;
        final Target target;

        public JarsVersionAndTarget(JkPathSequence jars, String version, Target target) {
            JkUtilsAssert.argument(jars != null, "jars cannot be null", new Object[0]);
            JkUtilsAssert.argument(version != null, "version cannot be null", new Object[0]);
            JkUtilsAssert.argument(target != null, "target cannot be null", new Object[0]);
            this.jars = jars;
            this.version = version;
            this.target = target;
        }
    }

    private static class Result {
        final boolean success;
        final List<String> params;

        public Result(boolean success, List<String> params) {
            this.success = success;
            this.params = params;
        }
    }

    static enum Target {
        JAVA,
        JS;

    }
}

